Rem FindChangeByList.vbs
Rem An InDesign VBScript
Rem
Rem Loads a series of tab-delimited strings from a text file, then performs a series
Rem of find/change operations based on the strings read from the file.
Rem
Rem The data file is tab-delimited, with carriage returns separating records.
Rem
Rem The format of each record in the file is:
Rem findType<tab>findProperties<tab>changeProperties<tab>findChangeOptions<tab>description
Rem 
Rem Where:
Rem <tab> is a tab character
Rem findType is "text", "grep", or "glyph" (this sets the type of find/change operation to use).
Rem findProperties is a properties record (as text) of the find preferences.
Rem changeProperties is a properties record (as text) of the change preferences.
Rem findChangeOptions is a properties record (as text) of the find/change options.
Rem description is a description of the find/change operation
Rem 
Rem Very simple example:
Rem text	{findWhat:"--"}	{changeTo:"^_"}	{includeFootnotes:true, includeMasterPages:true, includeHiddenLayers:true, wholeWord:false}	Find all double dashes and replace with an em dash.
Rem 
Rem More complex example:
Rem text	{findWhat:"^9^9.^9^9"}	{appliedCharacterStyle:"price"}	{include footnotes:true, include master pages:true, include hidden layers:true, whole word:false}	Find $10.00 to $99.99 and apply the character style "price".
Rem 
Rem All InDesign search metacharacters are allowed in the "find what" and "change to" properties.
Rem
Rem For more on InDesign/InCopy scripting see the documentation included in the Scripting SDK 
Rem available at http://www.adobe.com/devnet/indesign/sdk.html
Rem or visit the InDesign Scripting User to User forum at http://www.adobeforums.com
Rem
main
Function main()
	Set myInDesign = CreateObject("InDesign.Application.2024")
	Rem Set the user interaction level to allow the display of dialog boxes and alerts.
	myInDesign.ScriptPreferences.UserInteractionLevel = idUserInteractionLevels.idInteractWithAll
	If myInDesign.Documents.Count > 0 Then
	    If myInDesign.Documents.Item(1).Stories.Count > 0 Then
	        If myInDesign.Selection.Count > 0 Then
	            Select Case TypeName(myInDesign.Selection.Item(1))
	                Case "InsertionPoint", "Character", "Word", "TextStyleRange", "Line", "Paragraph", "TextColumn", "Text", "TextFrame", "Cell", "Table", "Row", "Column"
	                    myDisplayDialog(myInDesign)
	                Case Else:
	                    myFindChangeByList myInDesign, myInDesign.Documents.Item(1)
	            End Select
	     	Else
	            myFindChangeByList myInDesign, myInDesign.Documents.Item(1)
	        End If
	    Else
	        MsgBox ("The current document contains no text. Please open a document containing text and try again.")
	    End If
	Else
	    MsgBox ("No documents are open. Please open a document and try again.")
	End If
End Function
Function myDisplayDialog(myInDesign)
    myObject = "None"
    Set myDialog = myInDesign.Dialogs.Add
    myDialog.Name = "FindChangeByList"
    With myDialog.DialogColumns.Add
        With .DialogRows.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .StaticLabel = "Search Range"
                End With
            End With
            With .DialogColumns.Add
                Set myRangeButtons = .RadiobuttonGroups.Add
                With myRangeButtons
                    With .RadiobuttonControls.Add
                        .StaticLabel = "Document"
                        .CheckedState = True
                    End With
                    With .RadiobuttonControls.Add
                        .StaticLabel = "Selected Story"
                    End With
                    If myInDesign.Selection.Item(1).Contents <> "" Then
                        With .RadiobuttonControls.Add
                            .StaticLabel = "Selection"
	                        .CheckedState = True
                        End With
                    End If
                End With
            End With
        End With
    End With
    myResult = myDialog.Show
    If myResult = True Then
        Select Case myRangeButtons.SelectedButton
            Case 0:
                Set myObject = myInDesign.Documents.Item(1)
            Case 1:
                Select Case TypeName(myInDesign.Selection.Item(1))
                    Case "InsertionPoint", "Character", "Word", "TextStyleRange", "Line", "Paragraph", "TextColumn", "Text", "TextFrame"
                        Set myObject = myInDesign.Selection.Item(1).ParentStory
                End Select
            Case 2:
                Select Case TypeName(myInDesign.Selection.Item(1))
                    Case "Character", "Word", "TextStyleRange", "Line", "Paragraph", "TextColumn", "Text", "TextFrame"
                        If myInDesign.Selection.Item(1).Contents <> "" Then
                            Set myObject = myInDesign.Selection.Item(1).Texts.Item(1)
						Else
							Set myObject = myInDesign.Selection.Item(1).ParentStory
						End If
                Case Else:
                    Set myObject = Nothing
            End Select
        End Select
		myFindChangeByList myInDesign, myObject
    Else
        myDialog.Destroy
    End If
End Function
Function myFindChangeByList(myInDesign, myObject)
    myFileName = myFindFile(myInDesign, "\FindChangeSupport\FindChangeList.txt")
    If myFileName <> "" Then
        Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
        Set myFindChangeFile = myFileSystemObject.OpenTextFile(myFileName)
        Rem Loop through the find/change operations.
        Do While Not myFindChangeFile.AtEndOfStream
            myLine = myFindChangeFile.ReadLine
            Rem Ignore empty lines.
            If (Left(myLine, 4) = "text")Or(Left(myLine, 4) = "grep")Or(Left(myLine, 5) = "glyph") Then
                myFindChangeArray = Split(myLine, vbTab)
                Rem The first field in the line is the FindType string.
				myFindType = myFindChangeArray(0)
                Rem The second field in the line is the FindPreferences string.
                myFindPreferences = myFindChangeArray(1)
                Rem The third field in the line is the ChangePreferences string.
                myChangePreferences = myFindChangeArray(2)
                Rem The fourth field in the line can be either "once" or "all".
                Rem If it's "all", keep on searching until no instances of the
                Rem search text are found, if it's "once," search just once.
                myFindChangeOptions = myFindChangeArray(3)
                Select Case myFindType
                	Case "text"
                		myFindText myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions
                	Case "grep"
                		myFindGrep myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions
                	Case "glyph"
                		myFindGlyph myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions
                End Select
            End If
        Loop
        myFindChangeFile.Close
    End If
End Function
Function myFindText(myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions)
	Rem Clear the FindText/ChangeText Preferences before each find/change operation.
	myInDesign.FindTextPreferences = idNothingEnum.idNothing
	myInDesign.ChangeTextPreferences = idNothingEnum.idNothing
	myScript = "app.findTextPreferences.properties = " & myFindPreferences & ";" & vbCr
	myScript = myScript & "app.changeTextPreferences.properties = " & myChangePreferences & ";" & vbCr
	myScript = myScript & "app.findChangeTextOptions.properties = " & myFindChangeOptions & ";" & vbCr
    myInDesign.DoScript myScript, idScriptLanguage.idJavascript
	Set myFoundItems = myObject.ChangeText
	Rem Clear the FindText/ChangeText Preferences after each find/change operation.
	myInDesign.FindTextPreferences = idNothingEnum.idNothing
	myInDesign.ChangeTextPreferences = idNothingEnum.idNothing
End Function
Function myFindGrep(myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions)
	Rem Clear the FindGrep/ChangeGrep Preferences before each find/change operation.
	myInDesign.FindGrepPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGrepPreferences = idNothingEnum.idNothing
	myScript = "app.findGrepPreferences.properties = " & myFindPreferences & ";" & vbCr
	myScript = myScript & "app.changeGrepPreferences.properties = " & myChangePreferences & ";" & vbCr
	myScript = myScript & "app.findChangeGrepOptions.properties = " & myFindChangeOptions & ";" & vbCr
    myInDesign.DoScript myScript, idScriptLanguage.idJavascript
	Set myFoundItems = myObject.ChangeGrep
	Rem Clear the FindGrep/ChangeGrep Preferences after each find/change operation.
	myInDesign.FindGrepPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGrepPreferences = idNothingEnum.idNothing
End Function
Function myFindGlyph(myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions)
	Rem Clear the FindGlyph/ChangeGlyph Preferences before each find/change operation.
	myInDesign.FindGlyphPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGlyphPreferences = idNothingEnum.idNothing
	myScript = "app.findGlyphPreferences.properties = " & myFindPreferences & ";" & vbCr
	myScript = myScript & "app.changeGlyphPreferences.properties = " & myChangePreferences & ";" & vbCr
	myScript = myScript & "app.findChangeGlyphOptions.properties = " & myFindChangeOptions & ";" & vbCr
    myInDesign.DoScript myScript, idScriptLanguage.idJavascript
	Set myFoundItems = myObject.ChangeGlyph
	Rem Clear the FindGlyph/ChangeGlyph Preferences after each find/change operation.
	myInDesign.FindGlyphPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGlyphPreferences = idNothingEnum.idNothing
End Function 
Function myFindFile(myInDesign, myRelativePath)
	myFilePath = myInDesign.ActiveScript
	Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
	myFolderPath = myFileSystemObject.GetFile(myFilePath).ParentFolder.Path
	myFilePath = myFolderPath & myRelativePath
	If myFileSystemObject.FileExists(myFilePath) = True Then
		myFile = myFilePath
	Else
		myFile = myGetFileName(myFolderPath)
	End If
	myFindFile = myFile
End Function
Function myGetFileName(myFolder)
	Set myDialog = CreateObject("UserAccounts.CommonDialog")
	myDialog.Filter = "VBScript files|*.vbs"
	myDialog.InitialDir = myFolder
	myResult = myDialog.ShowOpen	 
	If myResult = 0 Then
	    myFileName = ""
	Else
		myFileName = myDialog.FileName
	End If
	myGetFileName = myFileName
End Function
'' SIG '' Begin signature block
'' SIG '' MIIpyQYJKoZIhvcNAQcCoIIpujCCKbYCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' 7nrgfY7c7HmYFiZbdiE7KmVkTYK/gerdR0igzg3ImPqg
'' SIG '' gg6WMIIGsDCCBJigAwIBAgIQCK1AsmDSnEyfXs2pvZOu
'' SIG '' 2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2Vy
'' SIG '' dCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAw
'' SIG '' WhcNMzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkq
'' SIG '' hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA1bQvQtAorXi3
'' SIG '' XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXrNCQH
'' SIG '' 4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAk
'' SIG '' ZLON4gh9NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXz
'' SIG '' ENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4RS6hnyzh
'' SIG '' GMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRX
'' SIG '' XegYE2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTCfMjq
'' SIG '' GzLmysL0p6MDDnSlrzm2q2AS4+jWufcx4dyt5Big2MEj
'' SIG '' R0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJa
'' SIG '' wv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tv
'' SIG '' k2E0XLyTRSiDNipmKF+wc86LJiUGsoPUXPYVGUztYuBe
'' SIG '' M/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eRGv7b
'' SIG '' UKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2
'' SIG '' FKZbS110YU0/EpF23r9Yy3IQKUHw1cVtJnZoEUETWJrc
'' SIG '' JisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa
'' SIG '' 6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27Iwy
'' SIG '' CQLMbDwMVhECAwEAAaOCAVkwggFVMBIGA1UdEwEB/wQI
'' SIG '' MAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7/mF7
'' SIG '' CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/n
'' SIG '' upiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAK
'' SIG '' BggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
'' SIG '' BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQw
'' SIG '' QwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
'' SIG '' aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
'' SIG '' cmwwHAYDVR0gBBUwEzAHBgVngQwBAzAIBgZngQwBBAEw
'' SIG '' DQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6mdNW
'' SIG '' 4AIapfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6j
'' SIG '' fCbVN7w6XUhtldU/SfQnuxaBRVD9nL22heB2fjdxyyL3
'' SIG '' WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzYgBoRGRjN
'' SIG '' YZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGV
'' SIG '' WEjVGv7XJz/9kNF2ht0csGBc8w2o7uCJob054ThO2m67
'' SIG '' Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3ZXQIU
'' SIG '' H0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m
'' SIG '' 6Ri+kAewQ3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL
'' SIG '' 6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJzy6kdJKt
'' SIG '' 2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFS
'' SIG '' i0HASIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8M4+u
'' SIG '' KIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm5yg/rBKupS8i
'' SIG '' bEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhR
'' SIG '' ILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+F
'' SIG '' SCH5Vzu0nAPthkX0tGFuv2jiJmCG6sivqf6UHedjGzqG
'' SIG '' VnhOMIIH3jCCBcagAwIBAgIQD99ExMbwCe2qSx8hp1o/
'' SIG '' EDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMB4XDTIzMTEw
'' SIG '' MzAwMDAwMFoXDTI1MTEwNDIzNTk1OVowgeYxEzARBgsr
'' SIG '' BgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMI
'' SIG '' RGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3JnYW5p
'' SIG '' emF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYDVQQG
'' SIG '' EwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNhbiBK
'' SIG '' b3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYDVQQL
'' SIG '' EyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29weSwg
'' SIG '' TXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCAiIwDQYJ
'' SIG '' KoZIhvcNAQEBBQADggIPADCCAgoCggIBAPCR5c/cgyg4
'' SIG '' MpJD46/+BWENPhE1RjN77VE2BWlIUdiO2nlWelKthEq9
'' SIG '' aK+1ofKIlDVc9mvxksuXBLESi7HBt3PCVBntNJkjBJd0
'' SIG '' EXeprmDK8HZrz8y/KSwEewbTMjneMRIRcJV2GMoAyUzK
'' SIG '' EyFJF7m6ByeoNBDDXMD4AQF5C2l3wfvRbREw+G+cyeya
'' SIG '' 564AHpFJ9sfqi9jAycnEZISxEf6N72GgGQr04cZt3JTb
'' SIG '' TktZGt6uWdZnseXDLGRHzFu4P4EqmVKvfMDliaKyWBzV
'' SIG '' lY+gmtE1vSxIK79WTySFrFY5j1vtbKyHybk6tv4TOmLt
'' SIG '' Lw4OP/ngVDiw/ZP+b7saGxPOEfuh5LOAr+llt+6V5NhT
'' SIG '' dkYPR6cbokawwX2DbGg7IeVuUPdf9bx/LznFV/8kPxCB
'' SIG '' J6qg0dqvCmQt/XbBXPJUvDoVJ9z+9hDlrAoLESqpHK8U
'' SIG '' 73UFGohBMJ/7itIyKngem7Ms98z6x0i63FJA8C06g+XE
'' SIG '' GJUmR28+c6ba+m/sc6xHI4zdL5a84UeRS9+u8UL8N4av
'' SIG '' fpSxHYTTxVjk1ZlwQ8dLARwEVWvTu4lRcEck3RPogx7o
'' SIG '' o8wUpSw9++m+Oi+K/z7RP/R67uCEtrkdM5N/xUoQR/t6
'' SIG '' 3P2ia9g8+LuYJ2BNBd1+qMcSoSlDfIFLFMeeqESbMv+j
'' SIG '' UsRQJn2lZvufAgMBAAGjggICMIIB/jAfBgNVHSMEGDAW
'' SIG '' gBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQU
'' SIG '' RPnrJ9yT26rMrd/Q4PfQSlgeaRcwPQYDVR0gBDYwNDAy
'' SIG '' BgVngQwBAzApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3
'' SIG '' LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/BAQDAgeA
'' SIG '' MBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0w
'' SIG '' gaowU6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
'' SIG '' bS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5nUlNB
'' SIG '' NDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRw
'' SIG '' Oi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVz
'' SIG '' dGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIx
'' SIG '' Q0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBc
'' SIG '' BggrBgEFBQcwAoZQaHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
'' SIG '' bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0T
'' SIG '' BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAPIZ9C4IuKg85
'' SIG '' 0H8M66OLMVllIpANmEjijIG/D65qP6jfma+2WFps2bb+
'' SIG '' vKKIEtG9TFtCAlSrA1OV6Ru78paq0gqw/q3QAN0OleFR
'' SIG '' F1NvT/TLO0ulhjXTETB1BwSKZaJH001DH0+60TmbGfVh
'' SIG '' mwLrEj2Zfl4COFwivZ/2tRoknZl5tmiu4hpjDbOapWd9
'' SIG '' 5cqliSoMeu2i588557eikdiDk6bXBa5MxHjyXvbvaIWN
'' SIG '' Zjsymn0L0sYkF3BToDN0h0XiTTZjZPHdSV2GjX6C/+Nj
'' SIG '' HSZGRtBaIxdAL1mKEWJV9qkEKELPb098rO+zGdn1E5IH
'' SIG '' pS1zSRIYngwZEjZfyoCkg1F+EW4QPs2+ff2CsW5i3IEd
'' SIG '' 7vL19w9oyce3S4wejvnlTKt+EQWn3gSPxrbkp3BxdWRN
'' SIG '' 25MKo5bS5bJ21BiwqcIJHcTxWNgaQPh4sZboi9IimCSn
'' SIG '' UenPQhqdeOuz0NnmepMEYd7L/kqnAeUZ8bW1UbITAWQ/
'' SIG '' UT3phjCfRQNpEh4OBoUcH8hOOHZ0Lh86mxv9lDP3i7c/
'' SIG '' bQzZnrJr2dS34JXolwREbTX9Yu5JB5X3YP539/M1Ms5O
'' SIG '' FZou/MHQBrWMsVBsEXDTWqwbMTYw/zvIiHXcGUnpzInw
'' SIG '' 1jQXlLrmwVHW4ug+A4LFO7bWC4mnZoDNWp0xnlGmbP6I
'' SIG '' ccJmW5C7RJMxghqLMIIahwIBATB9MGkxCzAJBgNVBAYT
'' SIG '' AlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8G
'' SIG '' A1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNp
'' SIG '' Z25pbmcgUlNBNDA5NiBTSEEzODQgMjAyMSBDQTECEA/f
'' SIG '' RMTG8AntqksfIadaPxAwDQYJYIZIAWUDBAIBBQCggZww
'' SIG '' GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
'' SIG '' BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcN
'' SIG '' AQkEMSIEIFHF78/7Mi2R2it42dn2XMZa78PihdMaEeQs
'' SIG '' WA+tQNtBMDAGCisGAQQBgjcCAQwxIjAgoB6AHABBAGQA
'' SIG '' bwBiAGUAIABJAG4ARABlAHMAaQBnAG4wDQYJKoZIhvcN
'' SIG '' AQEBBQAEggIAkcpyDGikd1ejSfg5ddtdNC4QQxegvYzD
'' SIG '' XdlfWU94a6Ut0LiYVlL2s07iLG/E2qRD55ttIWgQyA3g
'' SIG '' 6LG/HtvI+5NOtJUMYv1mDhxGO4fCU3f+OecZEhAiPNCR
'' SIG '' syCWlbHSpOpAJoajGReMXeYlKUxQ9/MnY/7WXRXwIciv
'' SIG '' tetvPrGvxV6FWFpdPErkHQbw+ci/WhlpTyIwGz3EXlea
'' SIG '' Jbm3wHQ6fV2/L7N4dshQL+TTELocuYyuVCV2BWQq2Hy5
'' SIG '' 8WqlaoQkR0iVNEBklzxCuS+qL9Jsm7HiSzlc9lxKJvJR
'' SIG '' OzJ1ZAP0EJNGS+WJu8e01h/Rj97LD7yTP0Oroweuw27J
'' SIG '' HsbjXPotEYGSQ2cNOBvW3+ENjub2/CFVVhovNjkYMnwr
'' SIG '' QKoNP0jB3/OftYhteRb8M16suN9siX6ffu3TkC6fczt9
'' SIG '' 9XIjFgIt/WbgoU2WTpDHBPdZk6HKLQ4uO19hcLAIhQBq
'' SIG '' cIah2TwHpRfDWQRNuJ5S3Rz0oWy5zO0/6RSxyy1hbHDn
'' SIG '' QTB/OyCIQqH7Ei5+cxREBqfH21opw2ZYFtQjM8ZPCWWN
'' SIG '' at0OSpdKBSbD7Ltn51TDhLB7rlGEsM6p97SkTUzFdwKC
'' SIG '' cyv2BumGGlinHq5vWmv2ZXvsxJbn7ov1ggm3lLM8/e1m
'' SIG '' nTONR8Gror0R5QIrz1nNVdy+iFQeN+IC9v+hghdAMIIX
'' SIG '' PAYKKwYBBAGCNwMDATGCFywwghcoBgkqhkiG9w0BBwKg
'' SIG '' ghcZMIIXFQIBAzEPMA0GCWCGSAFlAwQCAQUAMHgGCyqG
'' SIG '' SIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEwMTAN
'' SIG '' BglghkgBZQMEAgEFAAQggT5xjJwj4jC5QI44ioroCMwv
'' SIG '' eN1sq437DvGO7vjiZoICEQC1DWgc2Yi+Zotb0TXNuqss
'' SIG '' GA8yMDI0MDEyNjIzNTQ0MVqgghMJMIIGwjCCBKqgAwIB
'' SIG '' AgIQBUSv85SdCDmmv9s/X+VhFjANBgkqhkiG9w0BAQsF
'' SIG '' ADBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
'' SIG '' cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0
'' SIG '' ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5n
'' SIG '' IENBMB4XDTIzMDcxNDAwMDAwMFoXDTM0MTAxMzIzNTk1
'' SIG '' OVowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
'' SIG '' ZXJ0LCBJbmMuMSAwHgYDVQQDExdEaWdpQ2VydCBUaW1l
'' SIG '' c3RhbXAgMjAyMzCCAiIwDQYJKoZIhvcNAQEBBQADggIP
'' SIG '' ADCCAgoCggIBAKNTRYcdg45brD5UsyPgz5/X5dLnXaEO
'' SIG '' CdwvSKOXejsqnGfcYhVYwamTEafNqrJq3RApih5iY2nT
'' SIG '' WJw1cb86l+uUUI8cIOrHmjsvlmbjaedp/lvD1isgHMGX
'' SIG '' lLSlUIHyz8sHpjBoyoNC2vx/CSSUpIIa2mq62DvKXd4Z
'' SIG '' GIX7ReoNYWyd/nFexAaaPPDFLnkPG2ZS48jWPl/aQ9OE
'' SIG '' 9dDH9kgtXkV1lnX+3RChG4PBuOZSlbVH13gpOWvgeFmX
'' SIG '' 40QrStWVzu8IF+qCZE3/I+PKhu60pCFkcOvV5aDaY7Mu
'' SIG '' 6QXuqvYk9R28mxyyt1/f8O52fTGZZUdVnUokL6wrl76f
'' SIG '' 5P17cz4y7lI0+9S769SgLDSb495uZBkHNwGRDxy1Uc2q
'' SIG '' TGaDiGhiu7xBG3gZbeTZD+BYQfvYsSzhUa+0rRUGFOpi
'' SIG '' CBPTaR58ZE2dD9/O0V6MqqtQFcmzyrzXxDtoRKOlO0L9
'' SIG '' c33u3Qr/eTQQfqZcClhMAD6FaXXHg2TWdc2PEnZWpST6
'' SIG '' 18RrIbroHzSYLzrqawGw9/sqhux7UjipmAmhcbJsca8+
'' SIG '' uG+W1eEQE/5hRwqM/vC2x9XH3mwk8L9CgsqgcT2ckpME
'' SIG '' tGlwJw1Pt7U20clfCKRwo+wK8REuZODLIivK8SgTIUlR
'' SIG '' fgZm0zu++uuRONhRB8qUt+JQofM604qDy0B7AgMBAAGj
'' SIG '' ggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/
'' SIG '' BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNV
'' SIG '' HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYD
'' SIG '' VR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYD
'' SIG '' VR0OBBYEFKW27xPn783QZKHVVqllMaPe1eNJMFoGA1Ud
'' SIG '' HwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
'' SIG '' LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEy
'' SIG '' NTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEB
'' SIG '' BIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
'' SIG '' aWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9j
'' SIG '' YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
'' SIG '' ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5j
'' SIG '' cnQwDQYJKoZIhvcNAQELBQADggIBAIEa1t6gqbWYF7xw
'' SIG '' jU+KPGic2CX/yyzkzepdIpLsjCICqbjPgKjZ5+PF7SaC
'' SIG '' inEvGN1Ott5s1+FgnCvt7T1IjrhrunxdvcJhN2hJd6Pr
'' SIG '' kKoS1yeF844ektrCQDifXcigLiV4JZ0qBXqEKZi2V3mP
'' SIG '' 2yZWK7Dzp703DNiYdk9WuVLCtp04qYHnbUFcjGnRuSvE
'' SIG '' xnvPnPp44pMadqJpddNQ5EQSviANnqlE0PjlSXcIWiHF
'' SIG '' tM+YlRpUurm8wWkZus8W8oM3NG6wQSbd3lqXTzON1I13
'' SIG '' fXVFoaVYJmoDRd7ZULVQjK9WvUzF4UbFKNOt50MAcN7M
'' SIG '' mJ4ZiQPq1JE3701S88lgIcRWR+3aEUuMMsOI5ljitts+
'' SIG '' +V+wQtaP4xeR0arAVeOGv6wnLEHQmjNKqDbUuXKWfpd5
'' SIG '' OEhfysLcPTLfddY2Z1qJ+Panx+VPNTwAvb6cKmx5Adza
'' SIG '' ROY63jg7B145WPR8czFVoIARyxQMfq68/qTreWWqaNYi
'' SIG '' yjvrmoI1VygWy2nyMpqy0tg6uLFGhmu6F/3Ed2wVbK6r
'' SIG '' r3M66ElGt9V/zLY4wNjsHPW2obhDLN9OTH0eaHDAdwrU
'' SIG '' AuBcYLso/zjlUlrWrBciI0707NMX+1Br/wd3H3GXREHJ
'' SIG '' uEbTbDJ8WC9nR2XlG3O2mflrLAZG70Ee8PBf4NvZrZCA
'' SIG '' RK+AEEGKMIIGrjCCBJagAwIBAgIQBzY3tyRUfNhHrP0o
'' SIG '' ZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJV
'' SIG '' UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
'' SIG '' ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
'' SIG '' Q2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIwMzIzMDAw
'' SIG '' MDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJV
'' SIG '' UzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNV
'' SIG '' BAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
'' SIG '' SEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG
'' SIG '' 9w0BAQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFV
'' SIG '' xyUDxPKRN6mXUaHW0oPRnkyibaCwzIP5WvYRoUQVQl+k
'' SIG '' iPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITa
'' SIG '' EfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW
'' SIG '' 1OcoLevTsbV15x8GZY2UKdPZ7Gnf2ZCHRgB720RBidx8
'' SIG '' ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7
'' SIG '' QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRA
'' SIG '' p8ByxbpOH7G1WE15/tePc5OsLDnipUjW8LAxE6lXKZYn
'' SIG '' LvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCYOjgR
'' SIG '' s/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKV
'' SIG '' EStYdEAoq3NDzt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPV
'' SIG '' A+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6dSgkQe1C
'' SIG '' vwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHe
'' SIG '' k/45wPmyMKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtb
'' SIG '' iiKowSYI+RQQEgN9XyO7ZONj4KbhPvbCdLI/Hgl27Ktd
'' SIG '' RnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM
'' SIG '' 0jO0zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYB
'' SIG '' Af8CAQAwHQYDVR0OBBYEFLoW2W1NhS9zKXaaL3WMaiCP
'' SIG '' nshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiu
'' SIG '' HA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggr
'' SIG '' BgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUH
'' SIG '' MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
'' SIG '' BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYD
'' SIG '' VR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmww
'' SIG '' IAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcB
'' SIG '' MA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7ZvmKlEIgF+
'' SIG '' ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8
'' SIG '' acHPHQfpPmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExi
'' SIG '' HQwIgqgWvalWzxVzjQEiJc6VaT9Hd/tydBTX/6tPiix6
'' SIG '' q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAK
'' SIG '' fO+ovHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAn
'' SIG '' gkSumScbqyQeJsG33irr9p6xeZmBo1aGqwpFyd/EjaDn
'' SIG '' mPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR
'' SIG '' 8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un
'' SIG '' 8WbDQc1PtkCbISFA0LcTJM3cHXg65J6t5TRxktcma+Q4
'' SIG '' c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0d
'' SIG '' KNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm
'' SIG '' 8heZWcpw8De/mADfIBZPJ/tgZxahZrrdVcA6KYawmKAr
'' SIG '' 7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLeMt8E
'' SIG '' ifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP
'' SIG '' /JiW9lVUKx+A+sDyDivl1vupL0QVSucTDh3bNzgaoSv2
'' SIG '' 7dZ8/DCCBY0wggR1oAMCAQICEA6bGI750C3n79tQ4ghA
'' SIG '' GFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMx
'' SIG '' FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQ
'' SIG '' d3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNl
'' SIG '' cnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAw
'' SIG '' MDAwMFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMC
'' SIG '' VVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UE
'' SIG '' CxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGln
'' SIG '' aUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG
'' SIG '' 9w0BAQEFAAOCAg8AMIICCgKCAgEAv+aQc2jeu+RdSjww
'' SIG '' IjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuEDcQw
'' SIG '' H/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+
'' SIG '' vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBEEC7fgvMHhOZ0
'' SIG '' O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs06wXGXuxb
'' SIG '' Grzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW
'' SIG '' 7sp7XeOtyU9e5TXnMcvak17cjo+A2raRmECQecN4x7ax
'' SIG '' xLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtVgkEy19sEcypu
'' SIG '' kQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfd
'' SIG '' pCe8oU85tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6
'' SIG '' SPDgohIbZpp0yt5LHucOY67m1O+SkjqePdwA5EUlibaa
'' SIG '' RBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y
'' SIG '' 1YxwLEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOk
'' SIG '' GLimdwHhD5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXeeqxf
'' SIG '' jT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFr
'' SIG '' b7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZ
'' SIG '' qbId5RsCAwEAAaOCATowggE2MA8GA1UdEwEB/wQFMAMB
'' SIG '' Af8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiuHA9P
'' SIG '' MB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgP
'' SIG '' MA4GA1UdDwEB/wQEAwIBhjB5BggrBgEFBQcBAQRtMGsw
'' SIG '' JAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0
'' SIG '' LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMu
'' SIG '' ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
'' SIG '' dENBLmNydDBFBgNVHR8EPjA8MDqgOKA2hjRodHRwOi8v
'' SIG '' Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVk
'' SIG '' SURSb290Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADAN
'' SIG '' BgkqhkiG9w0BAQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22
'' SIG '' Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU
'' SIG '' 9BNKei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/5
'' SIG '' 9PesMHqai7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYDE3cn
'' SIG '' RNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4
'' SIG '' oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rb
'' SIG '' II01YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88nq2x2zm8j
'' SIG '' LfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNNn3O3
'' SIG '' AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UE
'' SIG '' BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTsw
'' SIG '' OQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQw
'' SIG '' OTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQBUSv85Sd
'' SIG '' CDmmv9s/X+VhFjANBglghkgBZQMEAgEFAKCB0TAaBgkq
'' SIG '' hkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcN
'' SIG '' AQkFMQ8XDTI0MDEyNjIzNTQ0MVowKwYLKoZIhvcNAQkQ
'' SIG '' AgwxHDAaMBgwFgQUZvArMsLCyQ+CXc6qisnGTxmcz0Aw
'' SIG '' LwYJKoZIhvcNAQkEMSIEIDQLoj+lrFWmr67FJWf/d16w
'' SIG '' OxStbT8yEziAw2IC8PweMDcGCyqGSIb3DQEJEAIvMSgw
'' SIG '' JjAkMCIEINL25G3tdCLM0dRAV2hBNm+CitpVmq4zFq9N
'' SIG '' GprUDHgoMA0GCSqGSIb3DQEBAQUABIICADJdObk5t9JW
'' SIG '' EAnOjTNgAfOZoVXMXrIhxndyYxYZ76cWZr5PtxePeoxH
'' SIG '' FEjvfw0ge3JCwbuPG3JJVCB6VSO2Vb9BpYikZd01irVx
'' SIG '' MnhCEyLiFwJKQIoCDk/BONLvM3Hm77jibFKN7m6IK4JS
'' SIG '' DrWGvFTeEAsC9hVpbTSOYSkhAYM0j0eqiAGpwHwtPGDJ
'' SIG '' sfc9qQVHgis7MaZOWUoKn+JKlzJRi8EsBAXyJPaXLN04
'' SIG '' k2x/ArdV+UboRoLNi8bdatLpU/JnPm47LNqBGvtdaw8B
'' SIG '' OlzeluaCuugApKlyZv+YCtBhFt3aojT7cir242vS4ocu
'' SIG '' sgljwp0nGsdrue52t3l55A4svXo3hRxl9vrmTWjIymj6
'' SIG '' gKU6r5LigtDSGWuqqcmgOdrAemXslB/tpcoLQ7ceRPIs
'' SIG '' 3Atba3aszrKXfVrt8JLT8JsL6jlvwxujF5D4KzvuZsY3
'' SIG '' mogRqhcFEFhNJad6tXV242zj3ZZ3h9EoCsMdbgCR0eFG
'' SIG '' 2d+uR5DuIs4bGE0PvKsbRgvLFG0KQQJ6M29fLsoreeqr
'' SIG '' Dv/XLjK7RpnzTE/eHQNaiZEyub1yTiFMCXgeIccBh0Ld
'' SIG '' xTdG24UTGHMxROYvGkEOQgf65Qxht/eRT6y2VrMl6YSZ
'' SIG '' 5MyM17KhiPFhxg5a/UmU2esXdaQawqFupenj2+I9/jck
'' SIG '' ASNMiciHERXC
'' SIG '' End signature block
